import SwiftUI

struct SurveyView: View {
    @StateObject private var viewModel = SurveyViewModel()
    @Environment(\.dismiss) private var dismiss
    
    var body: some View {
        NavigationStack {
            ZStack {
                Color.black.ignoresSafeArea()
                
                VStack {
                    // Progress indicator
                    ProgressView(value: viewModel.progress)
                        .progressViewStyle(LinearProgressViewStyle(tint: .green))
                        .padding(.horizontal)
                    
                    Text("Question \(viewModel.currentQuestionIndex + 1) of \(viewModel.totalQuestions)")
                        .font(.caption)
                        .foregroundColor(.secondary)
                        .padding(.bottom, 10)
                    
                    // Current question content
                    ScrollView {
                        VStack(spacing: 20) {
                            currentQuestionView
                        }
                        .padding()
                    }
                    
                    // Navigation buttons
                    HStack {
                        if viewModel.canGoBack {
                            Button("Back") {
                                viewModel.previousQuestion()
                            }
                            .buttonStyle(.bordered)
                            .foregroundColor(.blue)
                        }
                        
                        Spacer()
                        
                        Button(viewModel.isLastQuestion ? "Complete" : "Next") {
                            if viewModel.isLastQuestion {
                                viewModel.completeSurvey()
                                dismiss()
                            } else {
                                viewModel.nextQuestion()
                            }
                        }
                        .buttonStyle(.borderedProminent)
                        .disabled(!viewModel.canProceed)
                    }
                    .padding()
                }
            }
        }
        .navigationBarHidden(true)
        .onAppear {
            viewModel.startSurvey()
        }
    }
    
    @ViewBuilder
    private var currentQuestionView: some View {
        switch viewModel.currentQuestion {
        case .smoking:
            SmokingQuestionView(
                smokingStatus: $viewModel.surveyData.smokingStatus,
                cigarettesPerDay: $viewModel.surveyData.cigarettesPerDay
            )
        case .vaping:
            VapingQuestionView(
                vapingStatus: $viewModel.surveyData.vapingStatus,
                puffsPerDay: $viewModel.surveyData.puffsPerDay
            )
        case .alcohol:
            AlcoholQuestionView(
                alcoholUnitsPerWeek: $viewModel.surveyData.alcoholUnitsPerWeek
            )
        case .diet:
            DietQuestionView(
                fastFoodMealsPerWeek: $viewModel.surveyData.fastFoodMealsPerWeek
            )
        case .exercise:
            ExerciseQuestionView(
                exerciseFrequency: $viewModel.surveyData.exerciseFrequency
            )
        }
    }
}

// MARK: - Individual Question Views

struct SmokingQuestionView: View {
    @Binding var smokingStatus: SmokingStatus
    @Binding var cigarettesPerDay: Int
    
    var body: some View {
        VStack(alignment: .leading, spacing: 15) {
            Text("Do you smoke cigarettes?")
                .font(.headline)
                .foregroundColor(.white)
            
            VStack(spacing: 10) {
                ForEach(SmokingStatus.allCases, id: \.self) { status in
                    Button(action: {
                        smokingStatus = status
                        if status != .current {
                            cigarettesPerDay = 0
                        }
                    }) {
                        HStack {
                            Image(systemName: smokingStatus == status ? "checkmark.circle.fill" : "circle")
                                .foregroundColor(smokingStatus == status ? .green : .gray)
                            Text(status.displayName)
                                .foregroundColor(.white)
                            Spacer()
                        }
                    }
                    .buttonStyle(PlainButtonStyle())
                }
            }
            
            if smokingStatus == .current {
                VStack(alignment: .leading, spacing: 10) {
                    Text("Cigarettes per day:")
                        .font(.subheadline)
                        .foregroundColor(.white)
                    
                    Stepper(value: $cigarettesPerDay, in: 0...100) {
                        Text("\(cigarettesPerDay) per day")
                            .foregroundColor(.white)
                    }
                }
            }
        }
    }
}

struct VapingQuestionView: View {
    @Binding var vapingStatus: VapingStatus
    @Binding var puffsPerDay: Int
    
    var body: some View {
        VStack(alignment: .leading, spacing: 15) {
            Text("Do you use e-cigarettes or vape?")
                .font(.headline)
                .foregroundColor(.white)
            
            VStack(spacing: 10) {
                ForEach(VapingStatus.allCases, id: \.self) { status in
                    Button(action: {
                        vapingStatus = status
                        if status != .current {
                            puffsPerDay = 0
                        }
                    }) {
                        HStack {
                            Image(systemName: vapingStatus == status ? "checkmark.circle.fill" : "circle")
                                .foregroundColor(vapingStatus == status ? .green : .gray)
                            Text(status.displayName)
                                .foregroundColor(.white)
                            Spacer()
                        }
                    }
                    .buttonStyle(PlainButtonStyle())
                }
            }
            
            if vapingStatus == .current {
                VStack(alignment: .leading, spacing: 10) {
                    Text("Puffs per day:")
                        .font(.subheadline)
                        .foregroundColor(.white)
                    
                    Stepper(value: $puffsPerDay, in: 0...1000, step: 10) {
                        Text("\(puffsPerDay) puffs")
                            .foregroundColor(.white)
                    }
                }
            }
        }
    }
}

struct AlcoholQuestionView: View {
    @Binding var alcoholUnitsPerWeek: Double
    
    var body: some View {
        VStack(alignment: .leading, spacing: 15) {
            Text("How many alcohol units do you drink per week?")
                .font(.headline)
                .foregroundColor(.white)
            
            Text("(1 unit = 1/2 pint beer, 1 glass wine, 1 shot spirits)")
                .font(.caption)
                .foregroundColor(.secondary)
            
            Stepper(value: $alcoholUnitsPerWeek, in: 0...50, step: 0.5) {
                Text("\(alcoholUnitsPerWeek, specifier: "%.1f") units")
                    .foregroundColor(.white)
            }
            
            // Visual guide
            VStack(alignment: .leading, spacing: 5) {
                Text("Guidelines:")
                    .font(.caption)
                    .foregroundColor(.secondary)
                Text("• Low risk: 0-14 units")
                    .font(.caption2)
                    .foregroundColor(.green)
                Text("• Moderate risk: 15-25 units")
                    .font(.caption2)
                    .foregroundColor(.orange)
                Text("• High risk: 25+ units")
                    .font(.caption2)
                    .foregroundColor(.red)
            }
        }
    }
}

struct DietQuestionView: View {
    @Binding var fastFoodMealsPerWeek: Int
    
    var body: some View {
        VStack(alignment: .leading, spacing: 15) {
            Text("How many fast food meals do you eat per week?")
                .font(.headline)
                .foregroundColor(.white)
            
            Text("(Fast food, takeout, heavily processed meals)")
                .font(.caption)
                .foregroundColor(.secondary)
            
            Stepper(value: $fastFoodMealsPerWeek, in: 0...21) {
                Text("\(fastFoodMealsPerWeek) meals")
                    .foregroundColor(.white)
            }
            
            // Visual guide
            VStack(alignment: .leading, spacing: 5) {
                Text("Guidelines:")
                    .font(.caption)
                    .foregroundColor(.secondary)
                Text("• Excellent: 0-1 meals")
                    .font(.caption2)
                    .foregroundColor(.green)
                Text("• Good: 2-4 meals")
                    .font(.caption2)
                    .foregroundColor(.yellow)
                Text("• Poor: 5+ meals")
                    .font(.caption2)
                    .foregroundColor(.red)
            }
        }
    }
}

struct ExerciseQuestionView: View {
    @Binding var exerciseFrequency: ExerciseFrequency
    
    var body: some View {
        VStack(alignment: .leading, spacing: 15) {
            Text("How often do you exercise?")
                .font(.headline)
                .foregroundColor(.white)
            
            Text("(Moderate to vigorous physical activity)")
                .font(.caption)
                .foregroundColor(.secondary)
            
            VStack(spacing: 10) {
                ForEach(ExerciseFrequency.allCases, id: \.self) { frequency in
                    Button(action: {
                        exerciseFrequency = frequency
                    }) {
                        HStack {
                            Image(systemName: exerciseFrequency == frequency ? "checkmark.circle.fill" : "circle")
                                .foregroundColor(exerciseFrequency == frequency ? .green : .gray)
                            Text(frequency.displayName)
                                .foregroundColor(.white)
                            Spacer()
                        }
                    }
                    .buttonStyle(PlainButtonStyle())
                }
            }
        }
    }
}

#Preview {
    SurveyView()
}